<?php
/**
 * Forwarder API untuk menerima data transaksi dari E-STRUK
 * dan meneruskannya ke WhatsApp + menyimpan ke database + generate QR code + generate URL struk
 */

require_once __DIR__ . '/phpqrcode/phpqrcode.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Database Configuration
$db_host = 'localhost';
$db_name = 'wdpdev_forwader';
$db_user = 'wdpdev_master';
$db_pass = '!Panda9980';

// QR Code Configuration
$qr_base_url = 'https://gatestruk.wdpdev.co/qrcodes';
$qr_storage_path = __DIR__ . '/qrcodes';

// STRUK URL Configuration - URL struk di-generate oleh forwarder
$struk_base_url = 'https://gatestruk.wdpdev.co'; // GANTI DENGAN DOMAIN STRUK ANDA
$struk_path = '/receipt'; // Path untuk receipt

// Logging function
function logMessage($message) {
    $logFile = __DIR__ . '/forwarder.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Database connection function
function getDBConnection() {
    global $db_host, $db_name, $db_user, $db_pass;
    
    try {
        $pdo = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    } catch (PDOException $e) {
        logMessage("Database connection failed: " . $e->getMessage());
        return null;
    }
}

// Function to generate QR code
// MODIFIED: Added $phone parameter
function generateQRCode($transactionId, $forwardedTransactionId, $phone) {
    global $qr_storage_path, $qr_base_url;
    
    try {
        if (!is_dir($qr_storage_path)) {
            mkdir($qr_storage_path, 0755, true);
        }
        
        // MODIFIED: Tidak lagi di-hash (SHA256 dihapus).
        // Data digabung: TransactionID + Separator + Phone
        $qrContent = $transactionId . '|' . $phone;
        
        $filename = 'qrcode_' . $forwardedTransactionId . '.png';
        $filepath = $qr_storage_path . '/' . $filename;
        $fileurl = $qr_base_url . '/' . $filename;
        
        // Generate QR dengan data mentah (raw text)
        QRcode::png($qrContent, $filepath, 'L', 8, 2);
        
        logMessage("QR code generated for transaction $transactionId: $filepath | Content: $qrContent");
        
        return [
            'hash' => $qrContent, // Key tetap 'hash' agar kompatibel dengan DB, tapi isinya sekarang Raw Text
            'path' => $filepath,
            'url' => $fileurl,
            'filename' => $filename
        ];
        
    } catch (Exception $e) {
        logMessage("QR code generation failed: " . $e->getMessage());
        return null;
    }
}

// Function to generate struk URL
function generateStrukURL($transactionId) {
    global $struk_base_url, $struk_path;
    
    $struk_url = $struk_base_url . $struk_path . '/' . $transactionId;
    logMessage("Generated struk URL for transaction $transactionId: $struk_url");
    
    return $struk_url;
}

// Function to format complete WhatsApp message with URL
function formatWhatsAppMessage($originalMessage, $transactionId) {
    $struk_url = generateStrukURL($transactionId);
    
    // Tambahkan URL struk di akhir pesan
    $completeMessage = $originalMessage . "\n\nKlik link ini untuk melihat struk online dan qr code Anda:\n" . $struk_url;
    
    logMessage("Formatted complete message for transaction $transactionId, length: " . strlen($completeMessage));
    
    return $completeMessage;
}

// Function to save transaction to database
function saveTransactionToDB($data, $wablasResponse, $pdo) {
    global $qr_storage_path, $qr_base_url;
    
    try {
        $pdo->beginTransaction();
        
        $phone = $data['phone'];
        $message = $data['message']; // Pesan ASLI dari Go (tanpa URL)
        $transactionId = $data['transaction_id'];
        $transactionDetails = $data['transaction_details'] ?? [];
        $timestamp = $data['timestamp'] ?? date('Y-m-d H:i:s');
        
        // --- UPDATE: Ambil Invoice Code ---
        $invoiceCode = $transactionDetails['invoice_code'] ?? ''; 
        // ----------------------------------

        // Format phone number
        $phone = preg_replace('/\D/', '', $phone);
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        // Extract Wablas response data
        $wablasMessageId = '';
        $wablasDeviceId = '';
        $messageStatus = 'pending';
        
        if (isset($wablasResponse['data']['messages'][0]['id'])) {
            $wablasMessageId = $wablasResponse['data']['messages'][0]['id'];
        }
        if (isset($wablasResponse['data']['device_id'])) {
            $wablasDeviceId = $wablasResponse['data']['device_id'];
        }
        if (isset($wablasResponse['data']['messages'][0]['status'])) {
            $messageStatus = $wablasResponse['data']['messages'][0]['status'];
        }
        
        // FORMAT ULANG PESAN DENGAN MENAMBAHKAN URL STRUK
        $completeMessage = formatWhatsAppMessage($message, $transactionId);
        
        // Insert main transaction dengan pesan LENGKAP (termasuk URL)
        // --- UPDATE: Tambahkan kolom invoice_code ---
        $stmt = $pdo->prepare("
            INSERT INTO forwarded_transactions 
            (original_transaction_id, invoice_code, customer_phone, customer_name, total_amount, cash, change_amount, user_name, message_text, message_status, wablas_message_id, wablas_device_id, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $transactionId,
            $invoiceCode, // <-- Masukkan Invoice Code
            $phone,
            $transactionDetails['customer_name'] ?? '',
            $transactionDetails['total_amount'] ?? 0,
            $transactionDetails['cash'] ?? 0,
            $transactionDetails['change_amount'] ?? 0,
            $transactionDetails['user_name'] ?? '',
            $completeMessage, // Simpan pesan LENGKAP dengan URL
            $messageStatus,
            $wablasMessageId,
            $wablasDeviceId,
            $timestamp
        ]);
        
        $forwardedTransactionId = $pdo->lastInsertId();
        
        // MODIFIED: Memanggil generateQRCode dengan parameter tambahan $phone
        $qrCode = generateQRCode($transactionId, $forwardedTransactionId, $phone);
        
        // Update transaction with QR code data
        if ($qrCode) {
            $updateStmt = $pdo->prepare("
                UPDATE forwarded_transactions 
                SET qr_code_hash = ?, qr_code_path = ?, qr_code_url = ?
                WHERE id = ?
            ");
            $updateStmt->execute([
                $qrCode['hash'],
                $qrCode['path'],
                $qrCode['url'],
                $forwardedTransactionId
            ]);
        }
        
        // Insert transaction items
        if (isset($transactionDetails['items']) && is_array($transactionDetails['items'])) {
            $itemStmt = $pdo->prepare("
                INSERT INTO forwarded_transaction_items 
                (forwarded_transaction_id, product_name, quantity, price, subtotal)
                VALUES (?, ?, ?, ?, ?)
            ");
            
            foreach ($transactionDetails['items'] as $item) {
                $itemStmt->execute([
                    $forwardedTransactionId,
                    $item['product_name'] ?? $item['ProductName'] ?? '',
                    $item['quantity'] ?? $item['Quantity'] ?? 0,
                    $item['price'] ?? $item['Price'] ?? 0,
                    $item['subtotal'] ?? $item['Subtotal'] ?? 0
                ]);
            }
        }
        
        $pdo->commit();
        
        $logMessage = "Transaction $transactionId saved to database with ID: $forwardedTransactionId";
        if ($qrCode) {
            $logMessage .= " | QR Content: " . $qrCode['hash'];
        }
        logMessage($logMessage);
        
        return [
            'forwarded_id' => $forwardedTransactionId,
            'qr_code' => $qrCode,
            'complete_message' => $completeMessage
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        logMessage("Error saving transaction to database: " . $e->getMessage());
        throw $e;
    }
}

// Get request body
$input = file_get_contents('php://input');
$data = json_decode($input, true);

logMessage("Received request: " . $input);

// Validate input
if (!$data) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => 'Invalid JSON data'
    ]);
    logMessage("Error: Invalid JSON data");
    exit();
}

// Required fields
$requiredFields = ['phone', 'message', 'transaction_id'];
foreach ($requiredFields as $field) {
    if (!isset($data[$field]) || empty($data[$field])) {
        http_response_code(400);
        echo json_encode([
            'status' => 'error',
            'message' => "Missing required field: $field"
        ]);
        logMessage("Error: Missing required field: $field");
        exit();
    }
}

// Database connection
$pdo = getDBConnection();

try {
    // Extract data
    $phone = $data['phone'];
    $originalMessage = $data['message']; // Pesan ASLI dari Go (tanpa URL)
    $transactionId = $data['transaction_id'];
    $transactionDetails = $data['transaction_details'] ?? [];
    
    // Format phone number
    $phone = preg_replace('/\D/', '', $phone);
    if (substr($phone, 0, 1) === '0') {
        $phone = '62' . substr($phone, 1);
    }
    if (substr($phone, 0, 2) !== '62') {
        $phone = '62' . $phone;
    }
    
    logMessage("Processing transaction $transactionId for phone: $phone");
    
    // Konfigurasi Wablas
    $wablasDomain = 'https://sby.wablas.com';
    $wablasToken = 'DdziSjvLXmDwuyr5b0TILf6rTDOaM8glOStpdHps9F1O3j4HT7AC7K0';
    $wablasSecret = 'cH3GFGjm';
    
    // FORMAT PESAN LENGKAP DENGAN URL STRUK
    $completeMessage = formatWhatsAppMessage($originalMessage, $transactionId);
    
    // Update data dengan pesan lengkap
    $data['message'] = $completeMessage;
    
    // Prepare payload for Wablas (dengan pesan LENGKAP termasuk URL)
    $wablasPayload = [
        'data' => [
            [
                'phone' => $phone,
                'message' => $completeMessage, // Gunakan pesan LENGKAP dengan URL
                'isGroup' => false
            ]
        ]
    ];
    
    // Send to Wablas API
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $wablasDomain . '/api/v2/send-message');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($wablasPayload));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: ' . $wablasToken . '.' . $wablasSecret,
        'User-Agent: E-STRUK-Forwarder/1.0'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    logMessage("Wablas API Response - Code: $httpCode, Response: $response");
    
    if ($httpCode !== 200) {
        throw new Exception("Wablas API returned HTTP $httpCode: $response");
    }
    
    $wablasResponse = json_decode($response, true);
    
    // Check if Wablas response indicates success
    if (!isset($wablasResponse['status']) || $wablasResponse['status'] !== true) {
        $errorMsg = $wablasResponse['message'] ?? 'Unknown error from Wablas';
        throw new Exception("Wablas API error: $errorMsg");
    }
    
    // Save to database if connection is available
    $dbResult = null;
    if ($pdo) {
        try {
            $dbResult = saveTransactionToDB($data, $wablasResponse, $pdo);
        } catch (Exception $dbError) {
            logMessage("Database save failed but continuing: " . $dbError->getMessage());
        }
    }
    
    // Prepare response
    $responseData = [
        'status' => 'success',
        'message' => 'Message sent successfully',
        'transaction_id' => $transactionId,
        'phone' => $phone,
        'wablas_response' => $wablasResponse,
        'struk_url' => generateStrukURL($transactionId) // Tambahkan struk URL di response
    ];
    
    // Add database info if available
    if ($dbResult) {
        $responseData['db_transaction_id'] = $dbResult['forwarded_id'];
        if ($dbResult['qr_code']) {
            $responseData['qr_code'] = [
                'hash' => $dbResult['qr_code']['hash'],
                'url' => $dbResult['qr_code']['url']
            ];
        }
    }
    
    // Success response
    http_response_code(200);
    echo json_encode($responseData);
    
    $logMessage = "Success: Message sent for transaction $transactionId to $phone";
    if ($dbResult) {
        $logMessage .= " (DB ID: " . $dbResult['forwarded_id'] . ")";
        if ($dbResult['qr_code']) {
            $logMessage .= " | QR Code Generated: " . $dbResult['qr_code']['hash'];
        }
    }
    logMessage($logMessage);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage(),
        'transaction_id' => $transactionId ?? 'unknown'
    ]);
    
    logMessage("Exception: " . $e->getMessage());
}

// Close database connection if exists
if ($pdo) {
    $pdo = null;
}