<?php
/**
 * Receipt Viewer untuk menampilkan struk transaksi
 * Endpoint: https://gatestruk.wdpdev.co/receipt.php?id=6
 */

header('Content-Type: text/html; charset=utf-8');

// Database Configuration
$db_host = 'localhost';
$db_name = 'wdpdev_forwader';
$db_user = 'wdpdev_master';
$db_pass = '!Panda9980';

// Logging function
function logMessage($message) {
    $logFile = __DIR__ . '/receipt.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Database connection function
function getDBConnection() {
    global $db_host, $db_name, $db_user, $db_pass;
    
    try {
        $pdo = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    } catch (PDOException $e) {
        logMessage("Database connection failed: " . $e->getMessage());
        return null;
    }
}

// Get transaction ID from URL parameter
$transaction_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($transaction_id <= 0) {
    http_response_code(400);
    echo "Invalid transaction ID";
    exit();
}

logMessage("Accessing receipt for transaction ID: $transaction_id");

// Database connection
$pdo = getDBConnection();
if (!$pdo) {
    http_response_code(500);
    echo "Database connection error";
    exit();
}

try {
    $stmt = $pdo->prepare("
        SELECT 
            id,
            original_transaction_id,
            invoice_code, 
            customer_phone,
            customer_name,
            total_amount,
            cash,
            change_amount,
            user_name,
            message_text,
            message_status,
            wablas_message_id,
            wablas_device_id,
            qr_code_hash,
            qr_code_url,
            created_at
        FROM forwarded_transactions 
        WHERE original_transaction_id = ?
    ");
    
    $stmt->execute([$transaction_id]);
    $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$transaction) {
        http_response_code(404);
        echo "Transaction not found";
        logMessage("Transaction $transaction_id not found in database");
        exit();
    }

    // Fallback jika invoice_code kosong (untuk data lama)
    $display_invoice = !empty($transaction['invoice_code']) ? $transaction['invoice_code'] : 'INV #' . $transaction['original_transaction_id'];
    
    // Get transaction items
    $itemsStmt = $pdo->prepare("
        SELECT product_name, quantity, price, subtotal
        FROM forwarded_transaction_items
        WHERE forwarded_transaction_id = ?
    ");
    
    $itemsStmt->execute([$transaction['id']]);
    $items = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    logMessage("Found transaction $transaction_id with " . count($items) . " items");
    
} catch (Exception $e) {
    http_response_code(500);
    echo "Database error: " . $e->getMessage();
    logMessage("Database error: " . $e->getMessage());
    exit();
}

// Format currency
function formatCurrency($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

// Format date
function formatDate($dateString) {
    return date('d M Y, H:i', strtotime($dateString));
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Struk <?= htmlspecialchars($display_invoice) ?> - E-STRUK</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#eff6ff',
                            100: '#dbeafe',
                            200: '#bfdbfe',
                            300: '#93c5fd',
                            400: '#60a5fa',
                            500: '#3b82f6',
                            600: '#2563eb',
                            700: '#1d4ed8',
                            800: '#1e40af',
                            900: '#1e3a8a',
                        }
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            font-family: 'Inter', sans-serif;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
            margin: 0;
            padding: 0;
        }
        
        /* Mobile: No shadow, full screen */
        .receipt-shadow {
            box-shadow: none;
        }
        
        /* Desktop: Add shadow and centering */
        @media (min-width: 640px) {
            .receipt-shadow {
                box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            }
        }

        /* Smooth transitions */
        button {
            transition: all 0.2s ease;
        }
        
        button:active {
            transform: scale(0.95);
        }

        /* PRINT STYLES */
        @media print {
            body {
                background: white !important;
                padding: 0 !important;
            }
            
            @page {
                margin: 8mm;
                size: auto;
            }
            
            .receipt-print {
                box-shadow: none !important;
                border: none !important;
                border-radius: 0 !important;
                max-width: 100% !important;
                width: 100% !important;
            }
            
            .no-print {
                display: none !important;
            }

            .receipt-print * {
                color: black !important;
            }

            .receipt-print > div {
                background: white !important;
            }

            .receipt-print {
                font-size: 12px;
            }
            
            .receipt-print h1 { 
                font-size: 24px; 
            }
            
            .receipt-print h2 { 
                font-size: 16px; 
            }
        }
        
        /* Mobile optimization */
        @media (max-width: 639px) {
            body {
                font-size: 14px;
            }
            
            .receipt-shadow {
                border-radius: 0 !important;
                margin: 0 !important;
                min-height: 100vh;
            }
        }
    </style>
</head>
<body class="bg-gradient-to-br from-blue-50 to-blue-100 min-h-screen">
    <div class="receipt-shadow receipt-print bg-white sm:rounded-2xl sm:max-w-md w-full sm:mx-auto overflow-hidden border-0 sm:border sm:border-blue-100 sm:my-4 md:my-8">
        <!-- Header -->
        <div class="bg-gradient-to-r from-blue-600 to-blue-700 text-white p-4 sm:p-6 text-center">
            <div class="flex items-center justify-center mb-2 sm:mb-3">
                <i class="fas fa-receipt text-xl sm:text-2xl md:text-3xl mr-2 sm:mr-3"></i>
                <h1 class="text-xl sm:text-2xl md:text-3xl font-bold">E-STRUK</h1>
            </div>
            <p class="text-blue-100 font-medium text-xs sm:text-sm md:text-base">Digital Receipt</p>
            <div class="mt-3 sm:mt-4 bg-blue-500 inline-block px-3 sm:px-4 py-1 sm:py-1.5 rounded-full text-xs sm:text-sm font-semibold">
                <?= htmlspecialchars($display_invoice) ?>
            </div>
        </div>
        
        <!-- Content -->
        <div class="p-3 sm:p-4 md:p-6">
            <!-- Transaction Info -->
            <div class="mb-4 sm:mb-5 md:mb-6">
                <div class="flex items-center mb-3 sm:mb-4">
                    <i class="fas fa-info-circle text-blue-500 mr-2 text-base sm:text-lg"></i>
                    <h2 class="text-sm sm:text-base md:text-lg font-semibold text-gray-800">Informasi Transaksi</h2>
                </div>
                
                <div class="space-y-2 sm:space-y-2.5 md:space-y-3">
                    <div class="flex justify-between items-center py-2 border-b border-blue-50">
                        <div class="flex items-center text-gray-600 min-w-0 flex-shrink">
                            <i class="fas fa-calendar-alt text-blue-400 mr-2 w-4 sm:w-5 text-xs sm:text-sm flex-shrink-0"></i>
                            <span class="font-medium text-xs sm:text-sm">Tanggal</span>
                        </div>
                        <span class="text-gray-800 font-semibold text-xs sm:text-sm ml-2 text-right"><?= formatDate($transaction['created_at']) ?></span>
                    </div>
                    
                    <div class="flex justify-between items-center py-2 border-b border-blue-50">
                        <div class="flex items-center text-gray-600 min-w-0 flex-shrink">
                            <i class="fas fa-user text-blue-400 mr-2 w-4 sm:w-5 text-xs sm:text-sm flex-shrink-0"></i>
                            <span class="font-medium text-xs sm:text-sm">Kasir</span>
                        </div>
                        <span class="text-gray-800 font-semibold text-xs sm:text-sm ml-2 text-right truncate max-w-[60%]"><?= htmlspecialchars($transaction['user_name']) ?></span>
                    </div>
                    
                    <div class="flex justify-between items-center py-2 border-b border-blue-50">
                        <div class="flex items-center text-gray-600 min-w-0 flex-shrink">
                            <i class="fas fa-phone text-blue-400 mr-2 w-4 sm:w-5 text-xs sm:text-sm flex-shrink-0"></i>
                            <span class="font-medium text-xs sm:text-sm">Telepon</span>
                        </div>
                        <span class="text-gray-800 font-semibold text-xs sm:text-sm ml-2 text-right"><?= htmlspecialchars($transaction['customer_phone']) ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Items -->
            <div class="mb-4 sm:mb-5 md:mb-6">
                <div class="flex items-center mb-3 sm:mb-4">
                    <i class="fas fa-shopping-basket text-blue-500 mr-2 text-base sm:text-lg"></i>
                    <h2 class="text-sm sm:text-base md:text-lg font-semibold text-gray-800">Detail Item</h2>
                </div>
                
                <div class="bg-blue-50 rounded-lg p-2.5 sm:p-3 md:p-4">
                    <?php foreach ($items as $item): ?>
                    <div class="flex justify-between items-start py-2 sm:py-2.5 md:py-3 border-b border-blue-100 last:border-b-0 gap-2">
                        <div class="flex-1 min-w-0">
                            <div class="font-semibold text-gray-800 text-xs sm:text-sm break-words"><?= htmlspecialchars($item['product_name']) ?></div>
                            <div class="text-xs sm:text-sm text-blue-600 mt-0.5 sm:mt-1">
                                <?= $item['quantity'] ?> x <?= formatCurrency($item['price']) ?>
                            </div>
                        </div>
                        <div class="text-right flex-shrink-0 ml-2">
                            <div class="font-bold text-blue-700 text-xs sm:text-sm md:text-base whitespace-nowrap"><?= formatCurrency($item['subtotal']) ?></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Total -->
            <div class="mb-4 sm:mb-5 md:mb-6">
                <div class="bg-gradient-to-r from-blue-500 to-blue-600 rounded-xl p-3.5 sm:p-4 md:p-5 text-white">
                    <div class="flex justify-between items-center mb-2 sm:mb-3">
                        <span class="font-medium text-xs sm:text-sm md:text-base">Total</span>
                        <span class="font-bold text-base sm:text-lg md:text-xl"><?= formatCurrency($transaction['total_amount']) ?></span>
                    </div>
                    
                    <div class="border-t border-blue-400 mt-2.5 sm:mt-3 md:mt-4 pt-2.5 sm:pt-3 md:pt-4 flex justify-between items-center">
                        <span class="font-bold text-xs sm:text-sm md:text-base">GRAND TOTAL</span>
                        <span class="font-bold text-lg sm:text-xl md:text-2xl"><?= formatCurrency($transaction['total_amount']) ?></span>
                    </div>
                </div>
            </div>
            
            <!-- QR Code -->
            <?php if (!empty($transaction['qr_code_url'])): ?>
            <div class="mb-4 sm:mb-5 md:mb-6 text-center">
                <div class="flex items-center justify-center mb-3 sm:mb-4">
                    <i class="fas fa-qrcode text-blue-500 mr-2 text-base sm:text-lg"></i>
                    <h2 class="text-sm sm:text-base md:text-lg font-semibold text-gray-800">QR Code Verification</h2>
                </div>
                
                <div class="bg-white border-2 border-blue-200 rounded-xl p-2 sm:p-3 md:p-4 inline-block">
                    <img src="<?= $transaction['qr_code_url'] ?>" alt="QR Code" class="w-28 h-28 sm:w-40 sm:h-40 md:w-48 md:h-48 mx-auto rounded-lg">
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Footer -->
        <div class="bg-blue-50 border-t border-blue-200 p-3.5 sm:p-4 md:p-6 text-center no-print">
            <div class="flex items-center justify-center mb-2 sm:mb-3">
                <i class="fas fa-check-circle text-green-500 text-lg sm:text-xl mr-2"></i>
                <span class="text-green-600 font-semibold text-xs sm:text-sm md:text-base">Transaksi Berhasil</span>
            </div>
            <p class="text-blue-600 mb-1.5 sm:mb-2 text-xs sm:text-sm md:text-base">Terima kasih atas transaksi Anda</p>
            <p class="text-blue-500 text-xs sm:text-sm">E-STRUK</p>
            
            <div class="flex justify-center gap-2 sm:gap-3 md:gap-4 mt-3 sm:mt-4">
                <button onclick="window.print()" class="flex items-center justify-center px-4 sm:px-5 md:px-6 py-2 sm:py-2.5 bg-blue-500 hover:bg-blue-600 active:bg-blue-700 text-white rounded-lg transition duration-200 text-xs sm:text-sm md:text-base font-medium shadow-sm flex-1 sm:flex-none">
                    <i class="fas fa-print mr-1.5 sm:mr-2"></i>
                    <span>Print</span>
                </button>
                <button onclick="shareReceipt()" class="flex items-center justify-center px-4 sm:px-5 md:px-6 py-2 sm:py-2.5 bg-green-500 hover:bg-green-600 active:bg-green-700 text-white rounded-lg transition duration-200 text-xs sm:text-sm md:text-base font-medium shadow-sm flex-1 sm:flex-none">
                    <i class="fas fa-share-alt mr-1.5 sm:mr-2"></i>
                    <span>Share</span>
                </button>
            </div>
        </div>
    </div>

    <script>
        function shareReceipt() {
            var invoiceTitle = 'Struk <?= htmlspecialchars($display_invoice) ?> - E-STRUK';
            
            if (navigator.share) {
                navigator.share({
                    title: invoiceTitle,
                    text: 'Lihat struk digital transaksi saya di E-STRUK',
                    url: window.location.href
                })
                .then(() => console.log('Berbagi berhasil'))
                .catch((error) => console.log('Error berbagi', error));
            } else {
                navigator.clipboard.writeText(window.location.href).then(() => {
                    alert('Link struk telah disalin ke clipboard!');
                });
            }
        }

        // Animasi saat halaman dimuat
        document.addEventListener('DOMContentLoaded', function() {
            const receipt = document.querySelector('.receipt-shadow');
            receipt.style.transform = 'translateY(20px)';
            receipt.style.opacity = '0';
            
            setTimeout(() => {
                receipt.style.transition = 'all 0.5s ease-out';
                receipt.style.transform = 'translateY(0)';
                receipt.style.opacity = '1';
            }, 100);
        });
        
        // Prevent zoom on double tap
        let lastTouchEnd = 0;
        document.addEventListener('touchend', function (event) {
            const now = (new Date()).getTime();
            if (now - lastTouchEnd <= 300) {
                event.preventDefault();
            }
            lastTouchEnd = now;
        }, false);
    </script>
</body>
</html>